////////////////////////////////////////////////////////////////////////////////
// PIC USB HID Device								Create Date	2015/11/08
//	Header flie PIC16 / PIC18						Last Update	2018/04/04
// file name :  usbhiddev.h									Saka Softwares
// language  :  XC8 (v1.45)
////////////////////////////////////////////////////////////////////////////////
#ifndef _USB_HID_DEVICE_H
#define _USB_HID_DEVICE_H
// Include Files ///////////////////////////////////////////////////////////////
#include "usbconfig.h"

// Hardware Settings for PIC ///////////////////////////////////////////////////
#if defined(__16F1454) || defined(__16F1455) || defined(__16F1459) ||\
	  defined(__16LF1454) || defined(__16LF1455) || defined(__16LF1459)
	#define USB_RAM__ADDR	0x2000				// PIC USB RAM Address
	#define USBIF_FLG		PIR2bits.USBIF		// USB 全体割り込み
	#define USBIE_BIT		PIE2bits.USBIE		// USB 全体割り込み許可
#elif defined(__18F14K50)  || defined(__18F13K50) ||\
	defined(__18LF14K50) || defined(__18LF13K50)
	#define USB_RAM__ADDR	0x200				// PIC USB RAM Address
	#define USBIF_FLG		PIR2bits.USBIF		// USB 全体割り込み
	#define USBIE_BIT		PIE2bits.USBIE		// USB 全体割り込み許可
#elif defined(__18F24K50)  || defined(__18F25K50)  ||  defined(__18F45K50) ||\
	  defined(__18LF24K50) || defined(__18LF25K50) || defined(__18LF45K50)
	#define USB_RAM__ADDR	0x400				// PIC USB RAM Address
	#define USBIF_FLG		PIR3bits.USBIF		// USB 全体割り込み
	#define USBIE_BIT		PIE3bits.USBIE		// USB 全体割り込み許可
#elif defined(__18F2450) || defined(__18F2455) || defined(__18F2458) ||\
	  defined(__18F2550) || defined(__18F2553) ||\
	  defined(__18F4450) || defined(__18F4455) || defined(__18F4458) ||\
	  defined(__18F4550) || defined(__18F4553)
	#define USB_RAM__ADDR	0x400				// PIC USB RAM Address
	#define USBIF_FLG		PIR2bits.USBIF		// USB 全体割り込み
	#define USBIE_BIT		PIE2bits.USBIE		// USB 全体割り込み許可
#endif

////////////////////////////////////////////////////////////////////////////////
// UCFG用初期化パラメータ
#define _PPBM0		0x00			// Pingpong Buffer Mode 0 - ping pong bufferring disabled
#define _PPBM1		0x01			// Pingpong Buffer Mode 1 - ping pong on EP0 OUT only
#define _PPBM2		0x02			// Pingpong Buffer Mode 2 - ping pong on all endpoints
#define _LS			0x00			// Use Low-Speed USB Mode
#define _FS			0x04			// Use Full-Speed USB Mode
#define _TRINT		0x00			// Use internal transceiver
#define _TREXT		0x08			// Use external transceiver
#define _PUEN		0x10			// Use internal pull-up resistor
#define _OEMON		0x40			// Use SIE output indicator
#define _UTEYE		0x80			// Use Eye-Pattern test

// UEPn用初期化パラメータ
#define _CTRL		0x06			// Cfg Control pipe for this ep
#define _OUT		0x0C			// Cfg OUT only pipe for this ep
#define _IN			0x0A			// Cfg IN only pipe for this ep
#define _OUT_IN		0x0E			// Cfg both OUT & IN pipes for this ep
#define _HSHK		0x10			// Enable handshake packet

// Typedef /////////////////////////////////////////////////////////////////////
// コントロール転送 ステータス
typedef enum {
	WAIT_SETUP,						// セットアップパケット待ち
	CTRL_TRF_TX,					// コントロール送信中
	CTRL_TRF_RX,					// コントロール受信中
}CTRL_TRF_STAT;

////////////////////////////////////////////////////////////////////////////////
// USTSTレジスタのコピー
typedef union _USTATCOPY
{
	uint8_t Val;
	struct {
		unsigned		:1;
		unsigned PPBI	:1;			//0:偶数BDバンク / 1:奇数BDバンク
		unsigned DIR	:1;			//0:OUT or SETUP / 1:IN
		unsigned ENDP0	:1;			//エンドポイント bit0
		unsigned ENDP1	:1;			//エンドポイント bit1
		unsigned ENDP2	:1;			//エンドポイント bit2
		unsigned		:1;
    };
	struct {
		unsigned		:3;
		unsigned ENDP	:3;			//エンドポイント番号		
		unsigned		:1;
    };
} USTATCOPY;
#define PPBI_EVEN	0				//偶数BDバンク
#define PPBI_ODD	1				//奇数BDバンク
#define DIR_OUT		0				//受信
#define DIR_IN		1				//送信

////////////////////////////////////////////////////////////////////////////////
// BDT:バッファディスクリプタ テーブル
typedef union __BDT
{
	struct{
		union {
			uint8_t Val;
			struct {
				unsigned BC8:1;		//バイトカウント 8ビット
				unsigned BC9:1;		//バイトカウント 9ビット
				unsigned BSTALL:1;	//バッファストール イネーブルビット
				unsigned DTSEN:1;	//データトグル同期イネーブルビット
				unsigned :2;
				unsigned DTS:1;		//データトグル同期ビット 0:データ0 パケット 1:
				unsigned UOWN:1;	//USB オーナービット 0:CPU 1:SIE
			};
			struct {
				unsigned :2;
				unsigned PID0:1;	//パケットID 0 ビット
				unsigned PID1:1;	//パケットID 1 ビット
				unsigned PID2:1;	//パケットID 2 ビット
				unsigned PID3:1;	//パケットID 3 ビット
				unsigned :2;
			};
			struct {
				unsigned :2;
				unsigned PID:4;		//パケットID
				unsigned :2;
			};
		} STAT;						//ステータス レジスタ
		uint8_t CNT;				//バイトカウント レジスタ
		uint8_t ADRL;				//下位アドレスレジスタ
		uint8_t ADRH;				//上位アドレスレジスタ
	};
	struct {
		unsigned :8;
		unsigned :8;
		uint8_t* ADR;				//バッファアドレス ポインタ
	};
	uint32_t Val;					//BDT値
} BDT_ENTRY;
// バッファ ディスクリプタ ステータス レジスタ パラメータ
#define _BSTALL			0x04		// Buffer Stall enable
#define _DTSEN			0x08		// Data Toggle Synch enable
#define _DAT0			0x00		// DATA0 packet expected next
#define _DAT1			0x40		// DATA1 packet expected next
#define _DTSMASK		0x40		// DTS Mask
#define _USIE			0x80		// SIE owns buffer
#define _UCPU			0x00		// CPU owns buffer
// PID: トークン タイプ
#define SETUP_TOKEN		0b00001101	// コントロール セットアップ
#define OUT_TOKEN		0b00000001	// コントロール 受信
#define IN_TOKEN		0b00001001	// コントロール 送信

////////////////////////////////////////////////////////////////////////////////
// セットアップパケット
typedef union _CTRL_TRF_SETUP
{
	uint8_t dat[ EP0_BUFF_SIZE ];	//EP0のバッファ
	struct {
		// bmRequestType ///////////////////////////////////////////////////////
		union {
			uint8_t Val;			//バイト値
			struct {
				unsigned Recipient:5;		//受信側 0:デバイス 1:インターフェース 2:エンドポイント 3:その他
				unsigned RequestType:2;		//リクエスト タイプ 0:標準 1:クラス 2:ベンダー
				unsigned DataDir:1;			//転送方向 0:ホスト->デバイス 1:デバイス-.ホスト
			};
		} bmRequestType;			//リクエストタイプ

		// bRequest ////////////////////////////////////////////////////////////
		union {
			uint8_t Val;			//リクエスト
		}bRequest;					//特定のリクエス

		// wValue //////////////////////////////////////////////////////////////
		union {
			uint16_t Val;			//ワード値
			struct {
				uint8_t LB;			//下位
				uint8_t HB;			//上位
			};
			struct {
				uint8_t bFeature;	//Feature
				unsigned  :8;
			}FETR;					//Feature
			struct {
				uint8_t bDevAdr;	//デバイス アドレス
				unsigned  :8;
			}SADR;					//Set Address
			struct {
				uint8_t bCfgVal;	//コンフィグレーション値
				unsigned  :8;
			}SCFG;					//Set Configuration
			struct {
				uint8_t bIndx;		//インデックス
				uint8_t bType;		//タイプ
			}GDSR;					//Get Descriptor
			struct {
				uint8_t bAltID;		//アトリビュートID
				unsigned  :8;
			}INTF;					//Interface?
			struct {
				uint8_t bDuration;	//設定する期間
				uint8_t bReportID;	//レポートID
			}SIDL;					//Set Idlr時
			struct {
				uint8_t bProtocol;	//プロトコル 0:Boot Protocol / 1:Report Protocol
				unsigned  :8;
			}SPTL;					//Set Protocol時
		}wValue;					//bRequestが決める値

		// wIndex //////////////////////////////////////////////////////////////
		union {
			uint16_t Val;			//ワード値
			struct {
				uint8_t LB;			//下位
				uint8_t HB;			//上位
			};
			struct {
				unsigned  NUM:4;	//番号
				unsigned  :3;
				unsigned  DIR:1;	//方向 0:OUT / 1:IN
				unsigned  :8;
			}EP;					//エンドポイント
			struct{
				uint8_t bIntfID;	//インターフェースID
				unsigned  :8;
			}INTF;					//Interface時
		}wIndex;					//インデックスまたはオフセット地

		// wLength /////////////////////////////////////////////////////////////
		union { // struct -> union に修正 2016/03/02
			uint16_t Val;			//ワード値
			struct {
				uint8_t LB;			//下位
				uint8_t HB;			//上位
			};
		}wLength;					//データステージのバイト数
	};
} CTRL_TRF_SETUP;
// bmRequestType 値
#define RCPT_DEV		0
#define RCPT_INTF		1
#define RCPT_EP			2
#define RCPT_OTH		3

#define STANDARD		0
#define CLASS			1
#define VENDOR			2

#define HOST_TO_DEV		0
#define DEV_TO_HOST		1

// スタンダート リクエスト時の bRequest 値
#define GET_STATUS		0
#define CLR_FEATURE		1
#define SET_FEATURE		3
#define SET_ADR			5
#define GET_DSC			6
#define SET_DSC			7
#define GET_CFG			8
#define SET_CFG			9
#define GET_INTF		10
#define SET_INTF		11
#define SYNCH_FRAME		12

// Standard Feature Selectors
#define DEVICE_REMOTE_WAKEUP    0x01
#define ENDPOINT_HALT           0x00

////////////////////////////////////////////////////////////////////////////////
typedef union _CTRL_TRF_DATA
{
	uint8_t dat[ EP0_BUFF_SIZE ];
	struct {
		union {
			struct {
				unsigned  SPS:1;	//Self-Powered Status  0:Bus-Powered / 1:Self-Powered
				unsigned  RWP:1;	//RemoteWakeup  0:Disabled / 1:Enabled
				unsigned  :6;
			}DEV;
			struct {
				unsigned  HLT:1;	//Halt Status  0:Not Halted / 1:Halted
				unsigned  :7;
			}EP;
		};
		uint8_t :7;
    };
} CTRL_TRF_DATA;

////////////////////////////////////////////////////////////////////////////////
// 送信パイプ (inPipe)
typedef struct _IN_PIPE
{
	union {
		uint8_t		  *RAM;			//RAMの転送元アドレス
		const uint8_t *ROM;			//ROMの転送元アドレス
	}pSrc;							//転送元
	union {
		uint16_t	Val;			//値
		struct{
			uint8_t	LB;				//下位
			uint8_t	HB;				//上位
		};
	}wCount;						//転送数
	union {
		uint8_t Val;				//転送制御フラグ値
		struct{
			unsigned ROM:1;			//1:ROMから転送 0:RAMから転送
			unsigned NCPY:1;		//1:転送済み 0:転送
			unsigned ACK:1;			//1:返答送信 0:データ送信
			unsigned :4;
			unsigned BUSY:1;		//1:送信データ有り 0:送信データ無し
		};
	}flag;							//転送制御フラグ
} IN_PIPE;
//** パイプ フラグ値
#define PIPE_RAM	0b00000000		// 転送元/先が RAM
#define PIPE_ROM	0b00000001		// 転送元/先が ROM
#define PIPE_NCPY	0b00000010		// 転送不要
#define PIPE_ACK	0b00000100		// 返答送信 (2016/01/20修正)
#define PIPE_BUSY	0b10000000		// 送信中または受信中

////////////////////////////////////////////////////////////////////////////////
// USB状態フラグ
typedef enum {
	DETACHED_STATE,					// 未接続状態
	ATTACHED_STATE,					// 接続状態
	POWERED_STATE,					// USB電源投入状態
	DEFAULT_STATE,					// USB初期状態状態
	ADR_PENDING_STATE,				// ペンディンク状態
	ADDRESS_STATE,					// アドレス設定処理
	CONFIGURED_STATE,				// 設定完了状態 (通信中)
} USB_DEVICE_STATE;

////////////////////////////////////////////////////////////////////////////////
// 受信/送信 状態フラグ値
#define RCV_ERROR			0xFF	// 受信エラー
#define RCV_RECEING			0xF0	// 受信中
#define SND_ERROR			0xFF	// 送信エラー
#define SND_STANDBY			0xF0	// 送信待機

// Public Variable /////////////////////////////////////////////////////////////
USB_DEVICE_STATE USBDeviceState;	// USBデバイス状態遷移
#ifdef ENSBLE_SOF_COUNTER
	uint32_t USBSOFCounter;			// SOF用カウンタ
#endif

// usbhiddrv.c /////////////////////////////////////////////////////////////////
// USBのサービス関数です
void USBDeviceInit(void);
USB_DEVICE_STATE USBDeviceTasks(void);
void USBInterruptTasks(void);
bool	EPnOutIsBusy(uint8_t ep);
uint8_t	EPnOutReport(uint8_t ep, uint8_t* pbuf, int8_t len);
bool	EPnInIsBusy(uint8_t ep);
uint8_t	EPnInReport(uint8_t ep, uint8_t* pram, const uint8_t* prom, int8_t len, bool standby);
uint32_t USBGetSOFCounter(void);
uint32_t USBGetIdleTime(void);
uint8_t USBGetActiveProtocol(void);

#define HidRxIsBusy()		EPnOutIsBusy(1)
#define HidRxReport(a,b)	EPnOutReport(1,a,b)
#define HidTxIsBusy()		EPnInIsBusy(1)
#define HidTxRamReport(a,b)	EPnInReport(1,a,0,b,false)
#define HidTxRomReport(a,b)	EPnInReport(1,0,a,b,false)

#define USBGetDeviceState()		USBDeviceState
#define USBIsSuspended()		UCONbits.SUSPND
#define USBIsActive()			UIRbits.ACTVIF

#endif //_USB_HID_DEVICE_H
